import { NextResponse } from "next/server";
import { z } from "zod";
import { prisma } from "@/lib/db/prisma";
import { hashPassword } from "@/lib/auth/password";

const schema = z.object({
  email: z.string().email(),
  password: z.string().min(8),
  role: z.enum(["end_user_personal","end_user_business"]).default("end_user_personal"),
  fullName: z.string().optional()
});

export async function POST(req: Request) {
  const body = await req.json().catch(() => ({}));
  const parsed = schema.safeParse(body);
  if (!parsed.success) {
    return NextResponse.json({ error: "Invalid input", details: parsed.error.flatten() }, { status: 400 });
  }

  const { email, password, role, fullName } = parsed.data;

  const exists = await prisma.user.findUnique({ where: { email } });
  if (exists) return NextResponse.json({ error: "Email already in use" }, { status: 409 });

  const passwordHash = await hashPassword(password);

  const user = await prisma.user.create({
    data: { email, passwordHash, role, fullName }
  });

  return NextResponse.json({ user: { id: user.id, email: user.email, role: user.role } });
}
