import { NextResponse } from "next/server";
import { prisma } from "@/lib/db/prisma";
import { generateOrderNo } from "@/lib/utils/orderNo";
import { computePrice } from "@/lib/pricing/engine";
import { getSessionUser } from "@/lib/auth/session";
import { z } from "zod";

const schema = z.object({
  cartId: z.string().uuid(),
  tier: z.enum(["budget","standard","premium","luxury"]).default("standard"),
  depositPct: z.number().min(0).max(100).default(60)
});

export async function POST(req: Request) {
  const user = await getSessionUser(req);
  if (!user) return NextResponse.json({ error: "Unauthorized" }, { status: 401 });

  const body = await req.json().catch(() => ({}));
  const parsed = schema.safeParse(body);
  if (!parsed.success) return NextResponse.json({ error: "Invalid input" }, { status: 400 });

  const { cartId, tier, depositPct } = parsed.data;

  const cart = await prisma.cart.findUnique({
    where: { id: cartId },
    include: { items: { include: { product: true } } }
  });

  if (!cart || cart.userId !== user.id) return NextResponse.json({ error: "Cart not found" }, { status: 404 });
  if (cart.items.length === 0) return NextResponse.json({ error: "Cart is empty" }, { status: 400 });

  let subtotal = 0;
  for (const item of cart.items) {
    const p = item.product;
    const pricing = computePrice({
      baseCost: Number(p.baseCost),
      quantity: item.quantity,
      marginPct: tier === "budget" ? 10 : tier === "standard" ? 15 : tier === "premium" ? 20 : 25
    });
    subtotal += pricing.total;
  }

  const total = subtotal;
  const depositRequired = (total * depositPct) / 100;

  const order = await prisma.order.create({
    data: {
      orderNo: generateOrderNo(),
      userId: user.id,
      status: "created",
      tier,
      subtotal,
      total,
      depositRequired,
      currency: "BWP",
      items: {
        create: cart.items.map((it) => ({
          productId: it.productId,
          variantId: it.variantId,
          quantity: it.quantity,
          unitPrice: it.product.baseCost,
          isSample: it.isSample,
          brandingMethod: it.brandingMethod,
          notes: it.notes ?? null
        }))
      }
    },
    include: { items: true }
  });

  return NextResponse.json({ order });
}
